<?php

namespace App\Http\Controllers\Backend;

use Exception;
use App\Models\User;
use App\Models\Entity;
use App\Models\Direction;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;
use App\Http\Controllers\Controller;
use App\Models\CategorieUtilisateur;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Spatie\Permission\Models\Permission;
use App\Notifications\RegisterUserNotification;
use App\Notifications\Auth\AccountCreationNotification;
use App\Http\Requests\Auth\RegisterUserFromAdminRequest;

class UserController extends Controller
{
    public function index()
    {
        if (Auth::check()) {
            $users = User::all();
            $roles = Role::all();
            return view('backend.pages.users.list-of-users', compact('users', 'roles'));
        }
        return back()->with('warning', "Vous n'êtes pas connecté.");
    }

    public function add() 
    {
        if (Auth::check()) {
            $directions = Direction::all();
            $userCategories = CategorieUtilisateur::all();
            $roles = Role::all();
            return view('backend.pages.users.add-user', compact('roles', 'userCategories'));
        }
        return back()->with('warning', "Vous n'êtes pas connecté.");
    }

    public function store(RegisterUserFromAdminRequest $request)
    {
        try {
            $user = new User();
            
            $user->first_name = $request->first_name;
            $user->last_name = $request->last_name;
            $user->email = $request->email;
            $user->entity_id = $request->entity;
            $user->password = Hash::make($request->password, [
                'rounds' => 12,
            ]);
            $user->email_verified_at = now();

            if ($user->save()) {
                $user->assignRole($request->role_name);
                $user->notify(new AccountCreationNotification());
                return back()->with('success', 'Utilisateur ajouté avec succès.');
            }
        }   
        catch(Exception $e) {
            dd($e->getMessage());
        }
    }

    public function edit(User $user)
    {
        if (Auth::check()) {
            $user = User::findOrFail($user->user_id);
            return view('backend.pages.users.edit-user', compact('user'));
        }
        return back()->with('warning', "Vous n'êtes pas connecté.");
    }

    public function update(Request $request, User $user)
    {
        $request->validate([
            'first_name' => 'required',
            'last_name' => 'required',
            'email' => 'required|email',
        ]);
        
        $first_name = $request->first_name;
        $last_name = $request->last_name;
        $email = $request->email;
        $password = $request->password;
        

        // dd($user, $request->all());


        if ($request->password) {
            $request->validate([
                'password' => 'required|confirmed|min:6',
            ]);
        }

        try {

            if ($first_name != $user->first_name || $last_name != $user->last_name || $email != $user->email && !isset($request->password)) {
                $user->update([
                    'first_name' => $first_name,
                    'last_name' => $last_name,
                ]);
                return back()->with('success', "Information(s) de l'Utilisateur modifiée(s) avec succès.");
            }
            elseif ((isset($request->password) && !password_verify($password, $user->password)) || $first_name != $user->first_name || $last_name != $user->last_name || $email != $user->email) {
                $user->update([
                    'first_name' => $first_name,
                    'last_name' => $last_name,
                    'password' => Hash::make($password, [
                        'rounds' => 12,
                    ]),
                ]);
                return back()->with('success', "Information(s) de l'Utilisateur modifiée(s) avec succès.");
            }
            else {
                return back()->with('info', "Aucun changement n'a été apporté.");
            }
        }
        catch(Exception $e) {
            dd($e->getMessage());
        }
    }

    public function destroy(User $user)
    {
        $roles = $user->roles;
        try {
            $user = User::findOrFail($user->user_id);
            foreach ($roles as $role) {
                if (($role->name == 'super_admin')) {
                    return back()->with('error','Impossible de supprimer cet utilisateur, car Il a le rôle de Super Admin.');
                }
            }
            
            if ($user->delete()) {
                return redirect()->route('admin.user.list')->with('success', 'Utilisateur supprimé avec succès.');
            }
        }
        catch(Exception $e) {
            dd($e->getMessage());
        }
    }

    public function show(User $user)
    {
        if (Auth::check()) {
            $roles = Role::all();
            $permissions = Permission::all();
            return view('backend.pages.users.show', compact('user', 'roles', 'permissions'));
        }
        return back()->with('warning', "Vous n'êtes pas connecté.");
    }

    public function assignRole(Request $request, User $user)
    {
        $request->validate([
            'role_name' => 'required',
        ]);

        try {
            if ($user->hasRole($request->role_name)) {
                return back()->with('warning', "Le rôle '$request->role_name' est déjà assigné à cet Utilisateur.");
            }
            $user->assignRole($request->role_name);
            return back()->with('success', "Rôle assigné avec succès.");
        }
        catch(Exception $e) {
            dd($e->getMessage());
        }
    }

    public function removeRole(User $user, Role $role)
    {
        try {
            if ($user->email == 'admin@domain.com' && $role->name == 'super_admin') {
                return back()->with('error', "Le rôle '$role->name' ne peut pas être retiré à cet Utilisateur.");
            }
            if ($role->name == 'user') {
                return back()->with('error', "Chaque Utilisateur doit avoir au moins le rôle '$role->name'.");
            }
            if ($user->hasRole($role)) {
                $user->removeRole($role);
                return back()->with('success', "Rôle retiré avec succès.");
            }
            return back()->with('warning', "Cet Utilisateur n'a pas le rôle '$role->name'.");
        
        }
        catch(Exception $e) {
            dd($e->getMessage());
        }
    }

    public function grantPermission(Request $request, User $user)
    {
        $request->validate([
            'permission_name'=> 'required',
        ]);
        try {
            if ($user->hasPermissionTo($request->permission_name)) {
                return back()->with('warning', "La permission '$request->permission_name' est déjà accordée à cet Utilisateur.");
            }
            $user->givePermissionTo($request->permission_name);
            return back()->with('success', "Permission accordée avec succès.");
        }
        catch(Exception $e) {
            dd($e->getMessage());
        }
    }

    public function revokePermission(User $user, Permission $permission)
    {
        try {
            if ($user->hasPermissionTo($permission)) {
                $user->revokePermissionTo($permission);
                return back()->with('success', "Permission retirée avec succès.");
            }
            return back()->with('warning', "Cet Utilisateur n'a pas la permission '$permission->name'.");
        }
        catch(Exception $e) {
            dd($e->getMessage());
        }
    }
}
