<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Http\Requests\StorePromoteurRequest;
use App\Http\Requests\updatePromoteurRequest;
use App\Models\Promoteur;
use Exception;
use Illuminate\Support\Facades\Auth;

class PromoteurController extends Controller
{
    // Liste des promoteurs
    public function index()
    {
        $promoteurs = Promoteur::where('etat', 1)->paginate(10);
        return view('backend.pages.promoteurs.list', compact('promoteurs'));
    }

    // Formulaire pour créer un promoteur
    public function create()
    {
        return view('backend.pages.promoteurs.add');
    }

    // Enregistrer un promoteur
    public function store(StorePromoteurRequest $request)
    {
        try {
            // Créer un nouveau promoteur et remplir les champs
            $promoteur = new Promoteur();
            $promoteur->nom_entreprise = $request->nom_entreprise;
            $promoteur->user_id = Auth::id(); // Assignation automatique de l'utilisateur connecté
            $promoteur->save();
            
            return redirect()->route('promoteur.index')->with('success', 'Promoteur enregistré avec succès !');
        } catch (Exception $e) {
            // Gérer les exceptions et afficher un message d'erreur
            return redirect()->back()->with('error', "Une erreur est survenue lors de l'enregistrement du promoteur.");
        }
    }

    // Formulaire pour éditer un promoteur existant
    public function edit(Promoteur $promoteur)
    {
        // Vérifie si l'utilisateur est bien le propriétaire du promoteur
        if ($promoteur->user_id !== Auth::id()) {
            return redirect()->route('promoteur.index')->with('error', "Vous n'avez pas l'autorisation de modifier ce promoteur.");
        }

        return view('backend.pages.promoteurs.edit', compact('promoteur'));
    }

    // Mettre à jour un promoteur existant
    public function update(updatePromoteurRequest $request, Promoteur $promoteur)
    {
        try {
            // Vérifie si l'utilisateur est bien le propriétaire du promoteur
            if ($promoteur->user_id !== Auth::id()) {
                return redirect()->route('promoteur.index')->with('error', "Vous n'avez pas l'autorisation de modifier ce promoteur.");
            }

            // Mise à jour des champs du promoteur
            $promoteur->nom = $request->nom;
            $promoteur->prenom = $request->prenom;
            $promoteur->nom_entreprise = $request->nom_entreprise;
            $promoteur->telephone = $request->telephone;
            $promoteur->adresse = $request->adresse;
            $promoteur->save();

            return redirect()->route('promoteur.index')->with('success', 'Promoteur mis à jour avec succès !');
        } catch (Exception $e) {
            return redirect()->back()->with('error', "Une erreur est survenue lors de la mise à jour du promoteur.");
        }
    }

    // Supprimer un promoteur
    public function delete(Promoteur $promoteur)
    {
        try {
            // Vérifie si l'utilisateur est bien le propriétaire du promoteur
            if ($promoteur->user_id !== Auth::id()) {
                return redirect()->route('promoteur.index')->with('error', "Vous n'avez pas l'autorisation de supprimer ce promoteur.");
            }

            $promoteur->etat = 0;
            $promoteur->save();
            return redirect()->route('promoteur.index')->with('success', 'Promoteur supprimé avec succès !');
        } catch (Exception $e) {
            return redirect()->back()->with('error', "Une erreur est survenue lors de la suppression du promoteur.");
        }
    }
}
